<?php

namespace App\Http\Controllers\API\v1;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use Spatie\Activitylog\Facades\Activity;
use Laravel\Sanctum\PersonalAccessToken;


class AuthController extends Controller
{
    public function handleMeetingAuth(Request $request)
    {
        // Validate request
        $validated = $request->validate([
            'email' => 'required|email',
            'first_name' => 'required|string|max:25',
            'last_name' => 'required|string|max:25',
        ]);

        // Verify referring domain
        $referer = parse_url($request->header('referer'), PHP_URL_HOST);
        $allowedDomains = ['skweela.com', '*.skweela.com'];
        
        $isAllowed = collect($allowedDomains)->contains(function ($domain) use ($referer) {
            return Str::is($domain, $referer);
        });

        // if (!$isAllowed) {
        //     return response()->json([
        //         'success' => false,
        //         'message' => 'Unauthorized domain',
        //         'referer' => $referer
        //     ], 403);
        // }

        // Find or create user
        Activity::disableLogging();
        $password = Str::random(8);
        $username = Str::slug($validated['first_name'].$validated['last_name'].Str::random(2));
        
        $user = User::firstOrCreate(
            ['email' => $validated['email']],
            [
                'first_name' => $validated['first_name'],
                'last_name' => $validated['last_name'],
                'username' => $username,
                'password' => Hash::make($password),
                'status' => 'active',
            ]
        );
        
        // Create Sanctum token (this will be our single token)
        $sanctumToken = $user->createToken(
            'meeting-api-token',
            ['create-meeting', 'join-meeting'],
            null // No expiration
        )->plainTextToken;

        // Update api_token field with the same token value
        $user->api_token = $sanctumToken;
        $user->save();
        
        Activity::enableLogging();

        // Log registration activity
        Activity::causedBy($user)
            ->performedOn($user)
            ->withProperties([
                'ip' => $request->ip(),
                'referer' => $referer
            ])
            ->event('Cross-Site Registration')
            ->log('User registered from external site');

        return response()->json([
            'success' => true,
            'credentials' => [
                'email' => $user->email,
                'password' => $password,
                'username' => $username
            ],
            'api_token' => $sanctumToken, // Same as sanctum_token
            'user' => $user->only(['id', 'first_name', 'last_name', 'email', 'username'])
        ]);
    }

    public function autoLogin(Request $request)
    {
        $token = $request->query('token');
        if (!$token) {
            return redirect('/login')->with('error', 'Token not provided');
        }

        $accessToken = PersonalAccessToken::findToken($token);
        
        if (!$accessToken) {
            return redirect('/login')->with('error', 'Invalid token');
        }

        Auth::login($accessToken->tokenable);
        return redirect('/dashboard');
    }
}